"use client";
import { FormEvent, useEffect, useState } from "react";
import { useSession } from "next-auth/react";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";
import { useRouter } from "next/router";
import {
  ArrowLeft,
  Save,
  AlertCircle,
  CheckCircle2,
  Upload,
  X,
  Image as ImageIcon,
} from "lucide-react";
import { Dialog } from "@/components/Dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

type CreateResp =
  | {
      ok: true;
      task: any;
    }
  | { ok: false; error: string };

type Department = { id: string; name: string };
type LocationItem = { id: string; name: string; departmentId: string };
type ElementItem = { id: string; name: string; locationId: string };
type Technician = { 
  id: string; 
  name: string | null; 
  email: string;
  isExternal?: boolean;
  companyName?: string | null;
};

export default function NewTaskPage() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const { t } = useLanguage();

  const [title, setTitle] = useState("");
  const [description, setDescription] = useState("");
  const [assignedToId, setAssignedToId] = useState<string>("");
  const [dueDate, setDueDate] = useState<string>("");
  const [startDate, setStartDate] = useState<string>("");
  const [isRecurring, setIsRecurring] = useState(false);
  const [recurrencePattern, setRecurrencePattern] = useState<string>("");
  const [uploadedFiles, setUploadedFiles] = useState<File[]>([]);

  // Scope selection
  const [scopeType, setScopeType] = useState<
    "element" | "location" | "department" | "general" | ""
  >("");
  const [deps, setDeps] = useState<Department[]>([]);
  const [depId, setDepId] = useState<string>("");
  const [locations, setLocations] = useState<LocationItem[]>([]);
  const [locationId, setLocationId] = useState<string>("");
  const [selectedLocations, setSelectedLocations] = useState<string[]>([]);
  const [allLocationsSelected, setAllLocationsSelected] = useState(false);
  const [els, setEls] = useState<ElementItem[]>([]);
  const [elId, setElId] = useState<string>("");
  const [selectedElements, setSelectedElements] = useState<string[]>([]);
  const [allElementsSelected, setAllElementsSelected] = useState(false);
  const [technicians, setTechnicians] = useState<Technician[]>([]);

  const [loading, setLoading] = useState(false);
  const [result, setResult] = useState<CreateResp | null>(null);
  const [err, setErr] = useState<string | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);

  // Load technicians
  useEffect(() => {
    (async () => {
      try {
        const r = await fetch("/api/admin/users?role=technician");
        const j = await r.json();
        if (j.ok) {
          setTechnicians(j.users || []);
        }
      } catch (e: any) {
        console.error("Failed to load technicians:", e);
      }
    })();
  }, []);

  // Load departments
  useEffect(() => {
    (async () => {
      try {
        const r = await fetch("/api/departments");
        const j = await r.json();
        if (j.ok) {
          setDeps(j.items || []);
        }
      } catch (e: any) {
        setErr(String(e?.message || e));
      }
    })();
  }, []);

  // Load locations when department changes
  useEffect(() => {
    if (!depId) {
      setLocations([]);
      setLocationId("");
      setEls([]);
      setElId("");
      return;
    }
    (async () => {
      try {
        const r = await fetch(`/api/locations?departmentId=${depId}`);
        const j = await r.json();
        if (j.ok) {
          setLocations(j.items || []);
          setLocationId("");
        }
      } catch (e: any) {
        setErr(String(e?.message || e));
      }
    })();
  }, [depId]);

  // Load elements when location changes (for element/location level tasks)
  useEffect(() => {
    if (!locationId || (scopeType !== "element" && scopeType !== "location")) {
      if (scopeType !== "department") {
        setEls([]);
        setElId("");
      }
      return;
    }
    (async () => {
      try {
        const r = await fetch(`/api/elements?locationId=${locationId}`);
        const j = await r.json();
        if (j.ok) {
          setEls(j.items || []);
          setElId("");
        }
      } catch (e: any) {
        setErr(String(e?.message || e));
      }
    })();
  }, [locationId, scopeType]);

  // Load elements for department-level tasks (when locations are selected)
  useEffect(() => {
    if (scopeType !== "department" || !depId) {
      if (scopeType !== "location" && scopeType !== "element") {
        setEls([]);
      }
      return;
    }

    const targetLocationIds = allLocationsSelected
      ? locations.map((l) => l.id)
      : selectedLocations;

    if (targetLocationIds.length === 0 && !allLocationsSelected) {
      setEls([]);
      return;
    }

    (async () => {
      try {
        // Load elements from all target locations
        const allElements: ElementItem[] = [];
        for (const locId of targetLocationIds) {
          const r = await fetch(`/api/elements?locationId=${locId}`);
          const j = await r.json();
          if (j.ok) {
            allElements.push(...(j.items || []));
          }
        }
        setEls(allElements);
      } catch (e: any) {
        setErr(String(e?.message || e));
      }
    })();
  }, [depId, selectedLocations, allLocationsSelected, locations, scopeType]);

  // Compress and convert image to base64
  const compressImage = async (file: File): Promise<string> => {
    return new Promise((resolve, reject) => {
      if (!file.type.startsWith("image/")) {
        const reader = new FileReader();
        reader.onload = () => {
          if (typeof reader.result === "string") {
            resolve(reader.result);
          } else {
            reject(new Error("Failed to read file"));
          }
        };
        reader.onerror = reject;
        reader.readAsDataURL(file);
        return;
      }

      const reader = new FileReader();
      reader.onload = (e) => {
        const img = new Image();
        img.onload = () => {
          const maxDimension = 1280;
          let width = img.width;
          let height = img.height;

          if (width > maxDimension || height > maxDimension) {
            if (width > height) {
              height = (height / width) * maxDimension;
              width = maxDimension;
            } else {
              width = (width / height) * maxDimension;
              height = maxDimension;
            }
          }

          const canvas = document.createElement("canvas");
          canvas.width = width;
          canvas.height = height;
          const ctx = canvas.getContext("2d");
          if (!ctx) {
            reject(new Error("Failed to get canvas context"));
            return;
          }

          ctx.drawImage(img, 0, 0, width, height);
          const compressedDataUrl = canvas.toDataURL("image/jpeg", 0.7);
          resolve(compressedDataUrl);
        };
        img.onerror = reject;
        img.src = e.target?.result as string;
      };
      reader.onerror = reject;
      reader.readAsDataURL(file);
    });
  };

  const onSubmit = async (e: FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setResult(null);
    setErr(null);

    const fileAttachments: string[] = [];
    for (const file of uploadedFiles) {
      try {
        const base64 = await compressImage(file);
        fileAttachments.push(base64);
      } catch (error) {
        console.error("Error converting file to base64:", error);
      }
    }

    const body: any = {
      title,
      description,
      assignedToId: assignedToId || null,
      attachments: fileAttachments,
    };

    if (dueDate) {
      body.dueDate = new Date(dueDate).toISOString();
    }

    if (isRecurring && recurrencePattern) {
      body.isRecurring = true;
      body.recurrencePattern = recurrencePattern;
      if (startDate) {
        body.startDate = new Date(startDate).toISOString();
      }
    }

    // Scope handling
    if (scopeType && scopeType !== "general") {
      body.scopeType = scopeType;

      if (scopeType === "element") {
        if (!depId || !locationId || !elId) {
          setErr(
            "Please select Department, Location, and Element for element-level task"
          );
          setLoading(false);
          return;
        }
        body.departmentId = depId;
        body.locationId = locationId;
        body.elementId = elId;
      } else if (scopeType === "location") {
        if (!depId || !locationId) {
          setErr(
            "Please select Department and Location for location-level task"
          );
          setLoading(false);
          return;
        }
        body.departmentId = depId;
        body.locationId = locationId;
        if (allElementsSelected) {
          body.selectedElements = []; // Empty means all elements
        } else if (selectedElements.length > 0) {
          body.selectedElements = selectedElements;
        } else {
          setErr(
            "Please select elements or choose 'All elements' for location-level task"
          );
          setLoading(false);
          return;
        }
      } else if (scopeType === "department") {
        if (!depId) {
          setErr("Please select Department for department-level task");
          setLoading(false);
          return;
        }
        body.departmentId = depId;
        if (allLocationsSelected) {
          body.selectedLocations = []; // Empty means all locations
        } else if (selectedLocations.length > 0) {
          body.selectedLocations = selectedLocations;
        } else {
          setErr(
            "Please select locations or choose 'All locations' for department-level task"
          );
          setLoading(false);
          return;
        }
        if (allElementsSelected) {
          body.selectedElements = []; // Empty means all elements
        } else if (selectedElements.length > 0) {
          body.selectedElements = selectedElements;
        } else {
          setErr(
            "Please select elements or choose 'All elements' for department-level task"
          );
          setLoading(false);
          return;
        }
      }
    } else {
      // General task - no department/location/element required
      body.scopeType = null;
    }

    try {
      const resp = await fetch("/api/tasks", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(body),
      });
      const json = (await resp.json()) as CreateResp;
      setLoading(false);
      setResult(json);
      if (json.ok) {
        setDialogOpen(true);
      }
    } catch (e: any) {
      setLoading(false);
      setErr(String(e?.message || e));
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files) return;
    const newFiles = Array.from(files);
    setUploadedFiles((prev) => [...prev, ...newFiles]);
  };

  const removeFile = (index: number) => {
    setUploadedFiles((prev) => prev.filter((_, i) => i !== index));
  };

  const success = result && result.ok ? result : null;

  if (status === "unauthenticated") {
    return (
      <main className='max-w-3xl mx-auto px-4 py-6'>
        <p>You must be logged in to create tasks.</p>
        <Link href='/login'>
          <Button>Login</Button>
        </Link>
      </main>
    );
  }

  const role = (session?.user as any)?.role?.toUpperCase() || "";
  if (role !== "ADMIN") {
    return (
      <main className='max-w-3xl mx-auto px-4 py-6'>
        <p>Only administrators can create tasks.</p>
        <Link href='/tasks'>
          <Button variant='outline'>Back to Tasks</Button>
        </Link>
      </main>
    );
  }

  return (
    <main className='max-w-3xl mx-auto px-4 py-6'>
      <header className='flex items-center gap-3 mb-6'>
        <Button asChild variant='outline'>
          <Link href='/tasks'>
            <ArrowLeft className='w-4 h-4' />
            Back
          </Link>
        </Button>
        <h1 className='text-2xl font-bold'>Create Task</h1>
      </header>

      {err && (
        <div className='alert alert-error mb-4'>
          <AlertCircle />
          <div>
            <p className='font-semibold'>Error</p>
            <p className='text-sm opacity-90'>{err}</p>
          </div>
        </div>
      )}

      <form onSubmit={onSubmit} className='card p-5 space-y-4'>
        <div className='space-y-2'>
          <Label htmlFor='title'>Title *</Label>
          <Input
            id='title'
            value={title}
            onChange={(e) => setTitle(e.target.value)}
            required
            placeholder='Task title'
          />
        </div>

        <div className='space-y-2'>
          <Label htmlFor='description'>Description *</Label>
          <Textarea
            id='description'
            value={description}
            onChange={(e) => setDescription(e.target.value)}
            required
            rows={5}
            placeholder='Task description'
          />
        </div>

        <div className='space-y-2'>
          <Label htmlFor='assignedTo'>Assign To Technician (Optional)</Label>
          <Select
            value={assignedToId || "none"}
            onValueChange={(value) =>
              setAssignedToId(value === "none" ? "" : value)
            }
          >
            <SelectTrigger id='assignedTo'>
              <SelectValue placeholder='Select technician (optional)' />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value='none'>Not Assigned</SelectItem>
              {technicians.map((tech) => {
                const displayName = tech.name || tech.email;
                const externalTag = tech.isExternal ? " [External]" : "";
                const companyInfo = tech.companyName ? ` (${tech.companyName})` : "";
                return (
                  <SelectItem key={tech.id} value={tech.id}>
                    {displayName}{externalTag}{companyInfo}
                  </SelectItem>
                );
              })}
            </SelectContent>
          </Select>
        </div>

        <div className='grid sm:grid-cols-2 gap-4'>
          <div className='space-y-2'>
            <Label htmlFor='dueDate'>Due Date (Optional)</Label>
            <Input
              id='dueDate'
              type='datetime-local'
              value={dueDate}
              onChange={(e) => setDueDate(e.target.value)}
            />
          </div>

          <div className='space-y-2'>
            <Label htmlFor='isRecurring' className='flex items-center gap-2'>
              <input
                type='checkbox'
                id='isRecurring'
                checked={isRecurring}
                onChange={(e) => setIsRecurring(e.target.checked)}
                className='w-4 h-4'
              />
              Recurring Task
            </Label>
            {isRecurring && (
              <>
                <Select
                  value={recurrencePattern}
                  onValueChange={setRecurrencePattern}
                >
                  <SelectTrigger>
                    <SelectValue placeholder='Select pattern' />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value='daily'>Daily</SelectItem>
                    <SelectItem value='weekly'>Weekly</SelectItem>
                    <SelectItem value='monthly'>Monthly</SelectItem>
                    <SelectItem value='semestral'>Semestral (Twice a Year)</SelectItem>
                  </SelectContent>
                </Select>
                <div className='space-y-2 mt-2'>
                  <Label htmlFor='startDate'>Start Date (Optional)</Label>
                  <Input
                    id='startDate'
                    type='date'
                    value={startDate}
                    onChange={(e) => setStartDate(e.target.value)}
                    placeholder='When should recurrence start?'
                  />
                  <p className='text-xs text-gray-500'>
                    If not set, recurrence starts from today
                  </p>
                </div>
              </>
            )}
          </div>
        </div>

        {/* Scope Selection */}
        <div className='space-y-4 border-t pt-4'>
          <div className='space-y-2'>
            <Label htmlFor='scopeType'>Task Scope *</Label>
            <Select
              value={scopeType}
              onValueChange={(value) => {
                setScopeType(value as any);
                // Reset selections when scope changes
                setDepId("");
                setLocationId("");
                setElId("");
                setSelectedLocations([]);
                setSelectedElements([]);
                setAllLocationsSelected(false);
                setAllElementsSelected(false);
              }}
              required
            >
              <SelectTrigger id='scopeType'>
                <SelectValue placeholder='Select task scope' />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value='general'>
                  General Task (No specific location/element)
                </SelectItem>
                <SelectItem value='element'>
                  Element-Level (Specific equipment)
                </SelectItem>
                <SelectItem value='location'>
                  Location-Level (All or selected equipment in a location)
                </SelectItem>
                <SelectItem value='department'>
                  Department-Level (All or selected locations/elements in a
                  department)
                </SelectItem>
              </SelectContent>
            </Select>
          </div>

          {/* Element-Level Scope */}
          {scopeType === "element" && (
            <div className='grid sm:grid-cols-3 gap-4'>
              <div className='space-y-2'>
                <Label htmlFor='department'>Department *</Label>
                <Select value={depId} onValueChange={setDepId} required>
                  <SelectTrigger id='department'>
                    <SelectValue placeholder='Select department' />
                  </SelectTrigger>
                  <SelectContent>
                    {deps.map((d) => (
                      <SelectItem key={d.id} value={d.id}>
                        {d.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className='space-y-2'>
                <Label htmlFor='location'>Location *</Label>
                <Select
                  value={locationId || ""}
                  onValueChange={setLocationId}
                  disabled={!depId}
                  required
                >
                  <SelectTrigger id='location'>
                    <SelectValue
                      placeholder={
                        !depId ? "Select department first" : "Select location"
                      }
                    />
                  </SelectTrigger>
                  <SelectContent>
                    {locations.map((loc) => (
                      <SelectItem key={loc.id} value={loc.id}>
                        {loc.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className='space-y-2'>
                <Label htmlFor='element'>Equipment *</Label>
                <Select
                  value={elId || ""}
                  onValueChange={setElId}
                  disabled={!locationId}
                  required
                >
                  <SelectTrigger id='element'>
                    <SelectValue
                      placeholder={
                        !locationId
                          ? "Select location first"
                          : "Select equipment"
                      }
                    />
                  </SelectTrigger>
                  <SelectContent>
                    {els.map((el) => (
                      <SelectItem key={el.id} value={el.id}>
                        {el.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          )}

          {/* Location-Level Scope */}
          {scopeType === "location" && (
            <div className='space-y-4'>
              <div className='grid sm:grid-cols-2 gap-4'>
                <div className='space-y-2'>
                  <Label htmlFor='department'>Department *</Label>
                  <Select value={depId} onValueChange={setDepId} required>
                    <SelectTrigger id='department'>
                      <SelectValue placeholder='Select department' />
                    </SelectTrigger>
                    <SelectContent>
                      {deps.map((d) => (
                        <SelectItem key={d.id} value={d.id}>
                          {d.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div className='space-y-2'>
                  <Label htmlFor='location'>Location *</Label>
                  <Select
                    value={locationId || ""}
                    onValueChange={(value) => {
                      setLocationId(value);
                      setSelectedElements([]);
                      setAllElementsSelected(false);
                    }}
                    disabled={!depId}
                    required
                  >
                    <SelectTrigger id='location'>
                      <SelectValue
                        placeholder={
                          !depId ? "Select department first" : "Select location"
                        }
                      />
                    </SelectTrigger>
                    <SelectContent>
                      {locations.map((loc) => (
                        <SelectItem key={loc.id} value={loc.id}>
                          {loc.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </div>

              {locationId && (
                <div className='space-y-2'>
                  <Label>Elements to Check *</Label>
                  <div className='space-y-2'>
                    <label className='flex items-center gap-2 cursor-pointer hover:bg-gray-50 p-2 rounded'>
                      <input
                        type='checkbox'
                        checked={allElementsSelected}
                        onChange={(e) => {
                          setAllElementsSelected(e.target.checked);
                          if (e.target.checked) {
                            setSelectedElements([]);
                          }
                        }}
                        className='w-4 h-4 cursor-pointer'
                      />
                      <span className='font-medium'>
                        All elements in this location
                      </span>
                    </label>
                    {!allElementsSelected && (
                      <div className='border rounded-md p-3 max-h-60 overflow-y-auto bg-gray-50'>
                        {els.length === 0 ? (
                          <p className='text-sm text-gray-500 py-2'>
                            No elements found in this location
                          </p>
                        ) : (
                          els.map((el) => (
                            <label
                              key={el.id}
                              className='flex items-center gap-2 py-2 px-2 hover:bg-white rounded cursor-pointer'
                            >
                              <input
                                type='checkbox'
                                checked={selectedElements.includes(el.id)}
                                onChange={(e) => {
                                  if (e.target.checked) {
                                    setSelectedElements([
                                      ...selectedElements,
                                      el.id,
                                    ]);
                                  } else {
                                    setSelectedElements(
                                      selectedElements.filter(
                                        (id) => id !== el.id
                                      )
                                    );
                                  }
                                }}
                                className='w-4 h-4 cursor-pointer'
                              />
                              <span className='text-sm'>{el.name}</span>
                            </label>
                          ))
                        )}
                      </div>
                    )}
                    {!allElementsSelected && selectedElements.length > 0 && (
                      <p className='text-sm text-gray-600'>
                        {selectedElements.length} element
                        {selectedElements.length !== 1 ? "s" : ""} selected
                      </p>
                    )}
                  </div>
                </div>
              )}
            </div>
          )}

          {/* Department-Level Scope */}
          {scopeType === "department" && (
            <div className='space-y-4'>
              <div className='space-y-2'>
                <Label htmlFor='department'>Department *</Label>
                <Select
                  value={depId}
                  onValueChange={(value) => {
                    setDepId(value);
                    setSelectedLocations([]);
                    setSelectedElements([]);
                    setAllLocationsSelected(false);
                    setAllElementsSelected(false);
                  }}
                  required
                >
                  <SelectTrigger id='department'>
                    <SelectValue placeholder='Select department' />
                  </SelectTrigger>
                  <SelectContent>
                    {deps.map((d) => (
                      <SelectItem key={d.id} value={d.id}>
                        {d.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {depId && (
                <>
                  <div className='space-y-2'>
                    <Label>Locations to Include *</Label>
                    <div className='space-y-2'>
                      <label className='flex items-center gap-2 cursor-pointer hover:bg-gray-50 p-2 rounded'>
                        <input
                          type='checkbox'
                          checked={allLocationsSelected}
                          onChange={(e) => {
                            setAllLocationsSelected(e.target.checked);
                            if (e.target.checked) {
                              setSelectedLocations([]);
                            }
                          }}
                          className='w-4 h-4 cursor-pointer'
                        />
                        <span className='font-medium'>
                          All locations in this department
                        </span>
                      </label>
                      {!allLocationsSelected && (
                        <div className='border rounded-md p-3 max-h-60 overflow-y-auto bg-gray-50'>
                          {locations.length === 0 ? (
                            <p className='text-sm text-gray-500 py-2'>
                              No locations found in this department
                            </p>
                          ) : (
                            locations.map((loc) => (
                              <label
                                key={loc.id}
                                className='flex items-center gap-2 py-2 px-2 hover:bg-white rounded cursor-pointer'
                              >
                                <input
                                  type='checkbox'
                                  checked={selectedLocations.includes(loc.id)}
                                  onChange={(e) => {
                                    if (e.target.checked) {
                                      setSelectedLocations([
                                        ...selectedLocations,
                                        loc.id,
                                      ]);
                                    } else {
                                      setSelectedLocations(
                                        selectedLocations.filter(
                                          (id) => id !== loc.id
                                        )
                                      );
                                    }
                                  }}
                                  className='w-4 h-4 cursor-pointer'
                                />
                                <span className='text-sm'>{loc.name}</span>
                              </label>
                            ))
                          )}
                        </div>
                      )}
                      {!allLocationsSelected &&
                        selectedLocations.length > 0 && (
                          <p className='text-sm text-gray-600'>
                            {selectedLocations.length} location
                            {selectedLocations.length !== 1 ? "s" : ""} selected
                          </p>
                        )}
                    </div>
                  </div>

                  <div className='space-y-2'>
                    <Label>Elements to Check *</Label>
                    <div className='space-y-2'>
                      <label className='flex items-center gap-2 cursor-pointer hover:bg-gray-50 p-2 rounded'>
                        <input
                          type='checkbox'
                          checked={allElementsSelected}
                          onChange={(e) => {
                            setAllElementsSelected(e.target.checked);
                            if (e.target.checked) {
                              setSelectedElements([]);
                            }
                          }}
                          className='w-4 h-4 cursor-pointer'
                        />
                        <span className='font-medium'>
                          All elements in selected locations
                        </span>
                      </label>
                      {!allElementsSelected && (
                        <div className='border rounded-md p-3 max-h-60 overflow-y-auto bg-gray-50'>
                          {(() => {
                            // Get all elements from selected locations (or all locations if allLocationsSelected)
                            const targetLocationIds = allLocationsSelected
                              ? locations.map((l) => l.id)
                              : selectedLocations;
                            const availableElements = els.filter((el) =>
                              targetLocationIds.includes(el.locationId || "")
                            );
                            if (availableElements.length === 0) {
                              return (
                                <p className='text-sm text-gray-500 py-2'>
                                  {selectedLocations.length === 0 &&
                                  !allLocationsSelected
                                    ? "Select locations first"
                                    : "No elements found in selected locations"}
                                </p>
                              );
                            }
                            return availableElements.map((el) => (
                              <label
                                key={el.id}
                                className='flex items-center gap-2 py-2 px-2 hover:bg-white rounded cursor-pointer'
                              >
                                <input
                                  type='checkbox'
                                  checked={selectedElements.includes(el.id)}
                                  onChange={(e) => {
                                    if (e.target.checked) {
                                      setSelectedElements([
                                        ...selectedElements,
                                        el.id,
                                      ]);
                                    } else {
                                      setSelectedElements(
                                        selectedElements.filter(
                                          (id) => id !== el.id
                                        )
                                      );
                                    }
                                  }}
                                  className='w-4 h-4 cursor-pointer'
                                />
                                <span className='text-sm'>{el.name}</span>
                                <span className='text-xs text-gray-400 ml-auto'>
                                  {el.locationId &&
                                    locations.find(
                                      (l) => l.id === el.locationId
                                    )?.name}
                                </span>
                              </label>
                            ));
                          })()}
                        </div>
                      )}
                      {!allElementsSelected && selectedElements.length > 0 && (
                        <p className='text-sm text-gray-600'>
                          {selectedElements.length} element
                          {selectedElements.length !== 1 ? "s" : ""} selected
                        </p>
                      )}
                    </div>
                  </div>
                </>
              )}
            </div>
          )}
        </div>

        <div className='space-y-2'>
          <Label htmlFor='file-upload'>Attachments (Optional)</Label>
          <div className='border-2 border-dashed border-input rounded-md p-4'>
            <input
              type='file'
              id='file-upload'
              multiple
              accept='image/*'
              onChange={handleFileChange}
              className='hidden'
            />
            <Label
              htmlFor='file-upload'
              className='flex flex-col items-center justify-center cursor-pointer hover:opacity-80 transition'
            >
              <Upload className='w-8 h-8 text-muted-foreground mb-2' />
              <span className='text-sm text-muted-foreground'>
                Click to upload images
              </span>
            </Label>
          </div>

          {uploadedFiles.length > 0 && (
            <div className='grid grid-cols-2 sm:grid-cols-3 gap-3 mt-3'>
              {uploadedFiles.map((file, index) => (
                <div
                  key={index}
                  className='relative group border rounded-md overflow-hidden bg-card'
                >
                  {file.type.startsWith("image/") ? (
                    <img
                      src={URL.createObjectURL(file)}
                      alt={file.name}
                      className='w-full h-32 object-cover'
                    />
                  ) : (
                    <div className='w-full h-32 flex items-center justify-center bg-muted'>
                      <ImageIcon className='w-8 h-8 text-muted-foreground' />
                    </div>
                  )}
                  <button
                    type='button'
                    onClick={() => removeFile(index)}
                    className='absolute top-2 right-2 p-1.5 bg-red-500 hover:bg-red-600 text-white rounded-full shadow-lg z-10'
                  >
                    <X className='w-4 h-4' />
                  </button>
                  <div className='absolute bottom-0 left-0 right-0 bg-black/70 text-white text-xs p-2 truncate'>
                    {file.name}
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>

        <div className='pt-2 flex items-center gap-3'>
          <Button type='submit' disabled={loading}>
            {loading ? (
              <>
                <div className='w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin' />
                Creating...
              </>
            ) : (
              <>
                <Save className='w-4 h-4' />
                Create Task
              </>
            )}
          </Button>
          <Button asChild variant='outline'>
            <Link href='/tasks'>Cancel</Link>
          </Button>
        </div>
      </form>

      <Dialog
        open={dialogOpen && !!success}
        onClose={() => setDialogOpen(false)}
        title='Task Created'
        footer={
          <>
            <Button
              variant='outline'
              onClick={() => {
                setDialogOpen(false);
                router.push("/tasks");
              }}
            >
              Close
            </Button>
            {success?.ok && (
              <Button asChild>
                <Link href={`/tasks/${success.task.id}`}>View Task</Link>
              </Button>
            )}
          </>
        }
      >
        {success?.ok && (
          <div className='space-y-3'>
            <div className='flex items-center gap-2 text-green-600'>
              <CheckCircle2 className='w-5 h-5' />
              <span>Task created successfully!</span>
            </div>
            <p className='text-sm text-muted-foreground'>
              The task has been assigned to the selected technician.
            </p>
          </div>
        )}
      </Dialog>
    </main>
  );
}
